<?php

namespace App\Observers;

use App\Notifications\SystemActivityNotification;
use App\Transaction;
use App\User;
use Illuminate\Notifications\DatabaseNotification;

class TransactionObserver
{
    private function userHasRoleLike($user, string $base): bool
    {
        try {
            if (method_exists($user, 'getRoleNames')) {
                $baseLc = strtolower(trim($base));
                foreach ($user->getRoleNames() as $r) {
                    $name = strtolower(trim((string)$r));
                    if ($name === $baseLc || str_starts_with($name, $baseLc.'#')) {
                        return true;
                    }
                }
            } elseif (method_exists($user, 'hasRole')) {
                // Fallback exact check
                return $user->hasRole($base);
            }
        } catch (\Throwable $e) {}
        return false;
    }
    public function created(Transaction $tx): void
    {
        $business_id = (int) $tx->business_id;
        $actorId = auth()->id();
        $type = (string) $tx->type;
        $action = 'created';

        if ($type === 'purchase_requisition') {
            $title = 'Purchase Requisition Created';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was created.';
            $url   = action([\App\Http\Controllers\PurchaseRequisitionController::class, 'show'], [$tx->id]);
        } elseif ($type === 'purchase_order') {
            $title = 'Purchase Order Created';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was created.';
            $url   = action([\App\Http\Controllers\PurchaseOrderController::class, 'edit'], [$tx->id]);
        } elseif ($type === 'purchase') {
            $title = 'Purchase Created';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was created.';
            $url   = action([\App\Http\Controllers\PurchaseController::class, 'show'], [$tx->id]);
        } else {
            return; // Only notify for relevant types
        }

        // If PR is immediately for product creation, include extra roles
        $extraMeta = [];
        if ($type === 'purchase_requisition' && ($tx->status === 'for_product_creation')) {
            $extraMeta['status_to'] = 'for_product_creation';
            $action = 'for_product_creation';
        }
        $this->notifyRecipients($tx, $business_id, $actorId, $type, $action, $title, $body, $url, $extraMeta);
    }
    public function deleted(Transaction $tx): void
    {
        $business_id = (int) $tx->business_id;
        $actorId = auth()->id();
        $type = (string) $tx->type;
        $action = 'deleted';

        if ($type === 'purchase') {
            $title = 'Purchase Deleted';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was deleted.';
            $url   = action([\App\Http\Controllers\PurchaseController::class, 'index']);
        } elseif ($type === 'purchase_order') {
            $title = 'Purchase Order Deleted';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was deleted.';
            $url   = action([\App\Http\Controllers\PurchaseOrderController::class, 'index']);
        } elseif ($type === 'purchase_requisition') {
            $title = 'Purchase Requisition Deleted';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was deleted.';
            $url   = action([\App\Http\Controllers\PurchaseRequisitionController::class, 'index']);
        } elseif ($type === 'sell_transfer' || $type === 'purchase_transfer') {
            $title = 'Intercompany Sale Deleted';
            $body  = 'Ref ' . ($tx->ref_no ?? '') . ' was deleted.';
            $url   = action([\App\Http\Controllers\StockTransferController::class, 'index']);
        } else { // default to sale
            $title = 'Sale Deleted';
            $body  = 'Invoice ' . ($tx->invoice_no ?? '') . ' was deleted.';
            $url   = action([\App\Http\Controllers\SellController::class, 'index']);
        }

        $this->notifyRecipients($tx, $business_id, $actorId, $type, $action, $title, $body, $url);
    }
    public function updated(Transaction $tx): void
    {
        $dirty = array_keys($tx->getDirty());
        if (empty($dirty) || (count($dirty) === 1 && in_array('updated_at', $dirty, true))) {
            return;
        }

        $business_id = (int) $tx->business_id;
        $actorId = auth()->id();
        $type = (string) $tx->type;
        $action = 'updated';

        // Purchase Requisition moved to For Product Creation
        if (
            $type === 'purchase_requisition'
            && in_array('status', $dirty, true)
        ) {
            $to = (string) $tx->status;
            if ($to === 'for_product_creation') {
                $title = 'PR For Product Creation';
                $body = 'Ref ' . ($tx->ref_no ?? '') . ' is ready for product creation.';
                $url = action([\App\Http\Controllers\PurchaseRequisitionController::class, 'show'], [$tx->id]);
                $this->notifyRecipients(
                    $tx,
                    $business_id,
                    $actorId,
                    $type,
                    'for_product_creation',
                    $title,
                    $body,
                    $url,
                    ['status_to' => $to]
                );
                return;
            }
        }

        // Shipping-specific updates
        if (in_array('shipping_status', $dirty, true)) {
            $from = $tx->getOriginal('shipping_status') ?: 'Ordered';
            $to = $tx->shipping_status ?: 'N/A';
            $title = 'Shipping Updated';
            if ($type === 'sell') {
                $body = sprintf('%s to %s for Invoice %s', ucfirst($from), ucfirst($to), $tx->invoice_no ?? '');
                $url = action([\App\Http\Controllers\SellController::class, 'show'], [$tx->id]);
            } elseif ($type === 'purchase') {
                $body = sprintf('%s to %s for Ref %s', ucfirst($from), ucfirst($to), $tx->ref_no ?? '');
                $url = action([\App\Http\Controllers\PurchaseController::class, 'show'], [$tx->id]);
            } elseif ($type === 'sell_transfer' || $type === 'purchase_transfer') {
                $body = sprintf('%s to %s for Intercompany Ref %s', ucfirst($from), ucfirst($to), $tx->ref_no ?? '');
                $rootId = ($type === 'sell_transfer') ? $tx->id : ($tx->transfer_parent_id ?? $tx->id);
                $url = action([\App\Http\Controllers\StockTransferController::class, 'show'], [$rootId]);
            } else {
                return; // Unhandled
            }

            $this->notifyRecipients(
                $tx,
                $business_id,
                $actorId,
                $type,
                'shipping_updated',
                $title,
                $body,
                $url,
                ['shipping_from' => $from, 'shipping_to' => $to]
            );
            return;
        }

        // Generic updates
        if ($type === 'purchase') {
            $title = 'Purchase Updated';
            $body = 'Ref ' . ($tx->ref_no ?? '') . ' was updated.';
            $url = action([\App\Http\Controllers\PurchaseController::class, 'show'], [$tx->id]);
        } elseif ($type === 'purchase_order') {
            $title = 'Purchase Order Updated';
            $body = 'Ref ' . ($tx->ref_no ?? '') . ' was updated.';
            $url = action([\App\Http\Controllers\PurchaseOrderController::class, 'edit'], [$tx->id]);
        } elseif ($type === 'purchase_requisition') {
            $title = 'Purchase Requisition Updated';
            $body = 'Ref ' . ($tx->ref_no ?? '') . ' was updated.';
            $url = action([\App\Http\Controllers\PurchaseRequisitionController::class, 'index']);
        } elseif ($type === 'sell_transfer' || $type === 'purchase_transfer') {
            $title = 'Intercompany Sale Updated';
            $body = 'Ref ' . ($tx->ref_no ?? '') . ' was updated.';
            $rootId = ($type === 'sell_transfer') ? $tx->id : ($tx->transfer_parent_id ?? $tx->id);
            $url = action([\App\Http\Controllers\StockTransferController::class, 'show'], [$rootId]);
        } else { // default to sale
            $title = 'Sale Updated';
            $body = 'Invoice ' . ($tx->invoice_no ?? '') . ' was updated.';
            $url = action([\App\Http\Controllers\SellController::class, 'show'], [$tx->id]);
        }

        $this->notifyRecipients($tx, $business_id, $actorId, $type, $action, $title, $body, $url);
    }

    private function notifyRecipients(Transaction $tx, int $business_id, ?int $actorId, string $type, string $action, string $title, string $body, string $url, array $extraMeta = []): void
    {
        $payload = [
            'title' => $title,
            'body'  => $body,
            'url'   => $url,
            'meta'  => array_merge([
                'transaction_id' => $tx->id,
                'type' => $type,
                'action' => $action,
            ], $extraMeta),
        ];

        $recipientIds = [];
        $businessUsers = User::where('business_id', $business_id)->get();
        foreach ($businessUsers as $u) {
            if ($u->can('superadmin')) {
                $recipientIds[$u->id] = true;
            }
        }
        if ($actorId) {
            $recipientIds[$actorId] = true;
        }
        if (!empty($tx->created_by) && (int)$tx->created_by !== (int)$actorId) {
            $recipientIds[(int)$tx->created_by] = true;
        }
        if ($type === 'sell' && !empty($tx->commission_agent)) {
            $ids = [];
            $ca = $tx->commission_agent;
            if (is_numeric($ca)) {
                $ids[] = (int)$ca;
            } elseif (is_array($ca)) {
                $ids = array_map('intval', $ca);
            } elseif (is_string($ca)) {
                foreach (preg_split('/[,;\s]+/', $ca) as $piece) {
                    if (is_numeric($piece)) $ids[] = (int)$piece;
                }
            }
            foreach ($ids as $cid) if ($cid > 0) $recipientIds[$cid] = true;
        }
        if ($type === 'purchase' && !empty($tx->contact_id)) {
            $supplierUsers = User::where('business_id', $business_id)
                ->where('supplier_contact_id', (int)$tx->contact_id)
                ->get();
            foreach ($supplierUsers as $su) {
                $recipientIds[(int)$su->id] = true;
            }
        }

        // Also notify approvers for purchase order / requisition flows
        if (in_array($type, ['purchase_requisition', 'purchase_order'], true)) {
            foreach ($businessUsers as $u) {
                if (
                    ($type === 'purchase_requisition' && $u->can('purchase_requisition.approve')) ||
                    ($type === 'purchase_order' && $u->can('purchase_order.approve'))
                ) {
                    $recipientIds[(int)$u->id] = true;
                }
            }
        }

        // Always notify Purchasing Assistant for PR/PO/Purchase
        if (in_array($type, ['purchase_requisition', 'purchase_order', 'purchase'], true)) {
            foreach ($businessUsers as $u) {
                if ($this->userHasRoleLike($u, 'Purchasing Assistant')) {
                    $recipientIds[(int)$u->id] = true;
                }
            }
        }

        // If PR moved to For Product Creation, notify Encoder and Inventory Access roles
        if ($type === 'purchase_requisition' && $action === 'for_product_creation') {
            foreach ($businessUsers as $u) {
                if ($this->userHasRoleLike($u, 'Encoder') || $this->userHasRoleLike($u, 'Inventory Access')) {
                    $recipientIds[(int)$u->id] = true;
                }
            }
        }

        // On PR creation, also notify Encoders
        if ($type === 'purchase_requisition' && $action === 'created') {
            foreach ($businessUsers as $u) {
                if ($this->userHasRoleLike($u, 'Encoder')) {
                    $recipientIds[(int)$u->id] = true;
                }
            }
        }

        foreach (array_keys($recipientIds) as $uid) {
            $user = $businessUsers->firstWhere('id', (int)$uid) ?? User::find((int)$uid);
            if ($user && !$this->isDuplicate($user->id, $payload)) {
                $user->notify(new SystemActivityNotification($payload));
            }
        }
    }

    private function isDuplicate(int $userId, array $payload): bool
    {
        $since = now()->subMinutes(3);
        return DatabaseNotification::where('notifiable_id', $userId)
            ->where('type', SystemActivityNotification::class)
            ->where('created_at', '>=', $since)
            ->where(function ($q) use ($payload) {
                $q->where('data->meta->transaction_id', $payload['meta']['transaction_id'] ?? null)
                  ->where('data->meta->action', $payload['meta']['action'] ?? null);
                if (isset($payload['meta']['shipping_from'])) {
                    $q->where('data->meta->shipping_from', $payload['meta']['shipping_from']);
                }
                if (isset($payload['meta']['shipping_to'])) {
                    $q->where('data->meta->shipping_to', $payload['meta']['shipping_to']);
                }
            })
            ->exists();
    }
}
